document.addEventListener("DOMContentLoaded", function () {


    let enableToggle = document.getElementById("enableExtension");

    chrome.tabs.query({
        active: true,
        currentWindow: true
    }, function (tabs) {
        let currentDomain = new URL(tabs[0].url).hostname;

        chrome.storage.sync.get("enabledSites", (data) => {
            let enabledSites = data.enabledSites || {};
            enableToggle.checked = !!enabledSites[currentDomain];
        });

        enableToggle.addEventListener("change", function () {
            chrome.storage.sync.get("enabledSites", (data) => {
                let enabledSites = data.enabledSites || {};
                if (enableToggle.checked) {
                    enabledSites[currentDomain] = true;
                } else {
                    delete enabledSites[currentDomain];
                }
                chrome.storage.sync.set({
                    enabledSites
                });

                chrome.tabs.sendMessage(tabs[0].id, {
                    action: "toggleExtension",
                    enabled: enableToggle.checked
                });
            });
        });
    });

    // Tab Switching Functionality
    const tabs = document.querySelectorAll(".tab");
    const tabContents = document.querySelectorAll(".tab-content");

    // Function to switch tabs and store the active tab
    function switchTab(tab) {
        const target = tab.dataset.target;

        // Remove active class from all tabs and contents
        tabs.forEach(t => t.classList.remove("active"));
        tabContents.forEach(tc => tc.classList.remove("active"));

        // Add active class to the clicked tab and corresponding content
        tab.classList.add("active");
        document.getElementById(target).classList.add("active");

        // Store the active tab in Chrome storage
        chrome.storage.local.set({
            activeTab: target
        });
    }

    // Add event listeners to tabs
    tabs.forEach(tab => {
        tab.addEventListener("click", () => {
            switchTab(tab);
        });
    });

    // Restore the active tab when the popup opens
    chrome.storage.local.get("activeTab", function (data) {
        const savedTab = data.activeTab || "formTab"; // Default to 'formTab' if no tab is stored
        const defaultTab = document.querySelector(`.tab[data-target="${savedTab}"]`);
        if (defaultTab) {
            switchTab(defaultTab);
        }
    });

    let inputFields = document.getElementById("inputFields");
    let addMoreButton = document.getElementById("addMore");
    let saveFormButton = document.getElementById("saveForm");
    let importButton = document.getElementById("importProfile");
    let exportButton = document.getElementById("exportProfile");

    function loadStoredForms() {
        chrome.storage.sync.get("formData", (data) => {
            formList.innerHTML = "";
            let storedForms = data.formData || {};
            for (let formId in storedForms) {
                let tr = document.createElement("tr");

                let tdFormId = document.createElement("td");
                tdFormId.innerHTML = `<b>${formId}</b>`;
                tr.appendChild(tdFormId);

                let tdValues = document.createElement("td");
                tdValues.textContent = JSON.stringify(storedForms[formId]);
                tr.appendChild(tdValues);

                let tdActions = document.createElement("td");
                tdActions.innerHTML = `<button class="edit-btn" data-formid="${formId}">Edit</button> 
                                   <button class="remove-btn" data-formid="${formId}">Delete</button>`;
                tr.appendChild(tdActions);

                formList.appendChild(tr);

            }
        });
    }

    addMoreButton.addEventListener("click", () => {
        let div = document.createElement("div");
        div.classList.add("input-group");
        div.innerHTML = `
            <input type="text" class="elementId" placeholder="Enter element ID">
            <input type="text" class="elementValue" placeholder="Enter Value">
        `;
        inputFields.appendChild(div);
    });

    saveFormButton.addEventListener("click", () => {
        let formData = {};
        let inputGroups = document.querySelectorAll(".input-group");

        inputGroups.forEach(group => {
            let formId = group.querySelector(".elementId").value.trim();
            let value = group.querySelector(".elementValue").value.trim();
            if (formId && value) {
                formData[formId] = value;
            }
        });

        if (Object.keys(formData).length > 0) {
            chrome.storage.sync.get("formData", (data) => {
                let storedData = data.formData || {};
                Object.assign(storedData, formData);
                chrome.storage.sync.set({
                    formData: storedData
                }, () => {
                    loadStoredForms();

                    inputFields.innerHTML = "";
                });
            });
        } else {
            alert("Please enter valid data.");
        }
    });

    formList.addEventListener("click", (e) => {
        if (e.target.classList.contains("remove-btn")) {
            let formIdToRemove = e.target.dataset.formid;
            chrome.storage.sync.get("formData", (data) => {
                let storedData = data.formData || {};
                delete storedData[formIdToRemove];
                chrome.storage.sync.set({
                    formData: storedData
                }, loadStoredForms);
            });
        } else if (e.target.classList.contains("edit-btn")) {
            let formIdToEdit = e.target.dataset.formid;
            chrome.storage.sync.get("formData", (data) => {
                let storedData = data.formData || {};
                let newValue = prompt(`Edit value for ${formIdToEdit}:`, storedData[formIdToEdit]);
                if (newValue !== null) {
                    storedData[formIdToEdit] = newValue;
                    chrome.storage.sync.set({
                        formData: storedData
                    }, loadStoredForms);
                }
            });
        }
    });

    exportButton.addEventListener("click", () => {
        chrome.storage.sync.get("formData", (data) => {
            let storedData = JSON.stringify(data.formData, null, 2);
            let blob = new Blob([storedData], {
                type: "application/json"
            });
            let url = URL.createObjectURL(blob);
            let a = document.createElement("a");
            a.href = url;
            a.download = "form_profile.json";
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
        });
    });

    importButton.addEventListener("change", (event) => {
        let file = event.target.files[0];
        if (file) {
            let reader = new FileReader();
            reader.onload = function (e) {
                let importedData = JSON.parse(e.target.result);
                chrome.storage.sync.set({
                    formData: importedData
                }, () => {
                    alert("Profile imported successfully!");
                    loadStoredForms();
                });
            };
            reader.readAsText(file);
        }
    });

    loadStoredForms();

    const websiteList = document.getElementById("websiteList");
    const openWebsitesBtn = document.getElementById("openWebsites");

    const indexingTable = document.getElementById("indexingTable");
    const indexingResults = indexingTable.querySelector("tbody");

    // Restore saved URLs and indexing results
    chrome.storage.local.get(["savedWebsites", "indexingResults"], function (data) {
        if (data.savedWebsites && Array.isArray(data.savedWebsites)) {
            websiteList.value = data.savedWebsites.join("\n");
        }
        if (data.indexingResults && Array.isArray(data.indexingResults)) {
            indexingResults.innerHTML = data.indexingResults.join("");
        }
    });

    // Restore saved URLs when the popup is opened
    chrome.storage.local.get(["savedWebsites"], function (data) {
        if (data.savedWebsites && Array.isArray(data.savedWebsites)) {
            websiteList.value = data.savedWebsites.join("\n"); // Restore the URLs correctly
        }
    });

    // Save URLs when the user types (real-time saving)
    websiteList.addEventListener("input", function () {
        let websites = websiteList.value.trim().split("\n").map(site => site.trim()).filter(site => site);
        chrome.storage.local.set({ savedWebsites: websites });
    });

    // Open Multiple Websites Functionality
    openWebsitesBtn.addEventListener("click", () => {
        let websites = websiteList.value.trim().split("\n").map(site => site.trim()).filter(site => site);
        let selectedOption = document.querySelector('input[name="urlOption"]:checked').value;

        if (websites.length === 0) {
            alert("Please enter at least one website.");
            return;
        }

        // Save URLs before opening websites
        chrome.storage.local.set({ savedWebsites: websites }, function () {
            console.log("URLs saved successfully.");
        });

        websites.forEach(site => {
            let formattedURL = site.startsWith("http") ? site : `https://${site}`;
            if (selectedOption === "site") {
                formattedURL = `https://www.google.com/search?q=site:${site}`;
            }
            chrome.tabs.create({ url: formattedURL });
        });
    });

    chrome.storage.local.get(["extractedUrls"], function (data) {
        if (data.extractedUrls) {
            document.getElementById("extractedUrls").value = data.extractedUrls;
        }
    });

});

let websites = [];
let currentIndex = 0;
let isPaused = false;

async function checkIndexingStatus(site) {
    let googleSearchURL = `https://www.google.com/search?q=site:${site}`;

    try {
        let response = await fetch(googleSearchURL);

        if (response.status === 429) {
            isPaused = true; // Pause indexing
            alert("Google has detected too many requests. Please verify you are not a bot.");
            window.open(googleSearchURL, "_blank"); // Open the Google page for verification
            document.getElementById("resumeCheck").style.display = "block"; // Show resume button
            return {
                status: "Paused (Solve CAPTCHA)",
                reason: "Google blocked requests (429)."
            };
        }

        if (response.ok) {
            let pageText = await response.text();

            if (pageText.includes("did not match any documents")) {
                return {
                    status: "Not Indexed",
                    reason: "No results found in Google search."
                };
            } else {
                return {
                    status: "✅ Indexed",
                    reason: "Your website is indexed."
                };
            }
        } else {
            return {
                status: "❌ Error",
                reason: `HTTP Error: ${response.status}`
            };
        }
    } catch (error) {
        console.error("Check Indexing Error:", error);
        return {
            status: "Failed",
            reason: "Network issue or quota exceeded."
        };
    }
}

async function checkIndexing() {
    if (websites.length === 0) {
        alert("Please enter at least one website.");
        return;
    }

    let indexingTable = document.getElementById("indexingTable");
    let indexingHeader = document.getElementById("indexingHeader");
    let indexingResults = indexingTable.querySelector("tbody");

    indexingTable.style.display = "table";
    indexingHeader.style.display = "block";
    indexingResults.innerHTML = "";

    let resultsArray = [];

    for (; currentIndex < websites.length; currentIndex++) {
        if (isPaused) return; // Stop if CAPTCHA is required

        let site = websites[currentIndex];
        let row = document.createElement("tr");

        let link = document.createElement("a");
        link.href = `https://www.google.com/search?q=site:${site}`;
        link.target = "_blank"; // Open in new tab
        link.textContent = site;
        row.appendChild(link);

        let result = await checkIndexingStatus(site);

        let tdStatus = document.createElement("td");
        tdStatus.textContent = result.status;

        if (result.status === "Indexed") {
            tdStatus.style.color = "green";
        } else if (result.status === "Not Indexed") {
            tdStatus.style.color = "red";
        } else if (result.status.includes("Paused")) {
            tdStatus.style.color = "blue";
            return; // Stop checking until user resumes
        } else {
            tdStatus.style.color = "red";
        }

        row.appendChild(tdStatus);
        indexingResults.appendChild(row);

        resultsArray.push(row.outerHTML); // Save row for storage
    }
    chrome.storage.local.set({ indexingResults: resultsArray });
    indexingHeader.textContent = "✅Indexing Check Complete";
}

document.getElementById("checkIndexing").addEventListener("click", () => {
    websites = document.getElementById("websiteList").value.trim().split("\n").map(site => site.trim()).filter(site => site);
    currentIndex = 0;
    isPaused = false;
    document.getElementById("resumeCheck").style.display = "none"; // Hide resume button initially
    checkIndexing();
});

document.getElementById("resumeCheck").addEventListener("click", () => {
    isPaused = false;
    document.getElementById("resumeCheck").style.display = "none"; // Hide the button again
    checkIndexing();
});

// Extract URLs and store them
document.getElementById("extractUrls").addEventListener("click", () => {
    chrome.tabs.query({}, function (tabs) {
        let urls = tabs.map(tab => tab.url).join("\n");

        // Display URLs in textarea
        document.getElementById("extractedUrls").value = urls;

        // Store in Chrome storage
        chrome.storage.local.set({ extractedUrls: urls });
    });
});
    